#!/bin/sh
#
# /etc/rc.d/rc.S:  System initialization script.
#

### Early userspace

PATH=/sbin:/usr/sbin:/bin:/usr/bin

# Mount proc and sysfs if needed.
mount -at proc,sysfs

source /etc/rc.d/rc.common
source /lib/functions/have_emmc.sh
source /lib/functions/macaddr.sh
source /lib/functions/ubi.sh
stdio_syslog_start
navico_env_init

# Init udev
# Temporary hack for handling Stingray2's ramdisk, which
# populates serial ports in a manner udev doesn't like.
rm -f /dev/ttyS*
if [ -x /etc/rc.d/rc.udev ]; then
  /etc/rc.d/rc.udev start
fi

# Disable complex scheduling for internal solid state storage.
# (leave the CFQ default alone for connectable USB hard disks)
if [ -d /sys/block ]; then
   for BLOCKDEV in \
      $(ls -d /sys/block/mmcblk* /sys/block/mtdblock* /sys/block/ubiblk* /sys/block/loop* 2>/dev/null) ;
   do
      echo noop > ${BLOCKDEV}/queue/scheduler
   done
fi

# Start FUSE, if requested:
if [ -x /etc/rc.d/rc.fuse ]; then
  sh /etc/rc.d/rc.fuse start
fi


### Filesystem maintenance.

# Test to see if the root partition is read-only, like it ought to be.
READWRITE=no
if touch /fsrwtestfile 2>/dev/null; then
  rm -f /fsrwtestfile
  READWRITE=yes
else
  echo "Testing root filesystem status:  read-only filesystem"
fi

# See if a forced filesystem check was requested at shutdown:
if [ -r /etc/forcefsck ]; then
  FORCEFSCK="-f"
fi

# Check the root filesystem:
if [ ! $READWRITE = yes ]; then
  RETVAL=0
  if [ ! -r /etc/fastboot ]; then
    echo "Checking root filesystem:"
    /sbin/fsck $FORCEFSCK -C -a /
    RETVAL=$?
  fi
  if [ $RETVAL -eq 4 ]; then
    # fsck found errors that could not be fixed without human intervention.
    # For our embedded device we need to try really hard to make sure that
    # the system will always boot, so try more thorough checking in
    # non-interactive mode (assuming that all questions are answered yes)
    /sbin/fsck -f -C -y /
    RETVAL=$?
  fi

  # An error code of 2 or higher will require a reboot.
  if [ $RETVAL -ge 2 ]; then
    # An error code equal to or greater than 4 means that some errors
    # could not be corrected.
    if [ $RETVAL -ge 4 ]; then
      echo
      echo "***********************************************************"
      echo "*** An error occurred during the root filesystem check. ***"
      echo "*** You will now be given a chance to log into the      ***"
      echo "*** system in single-user mode to fix the problem.      ***"
      echo "***                                                     ***"
      echo "*** If you are using the ext2 filesystem, running       ***"
      echo "*** 'e2fsck -v -y <partition>' might help.              ***"
      echo "***********************************************************"
      echo
      echo "Once you exit the single-user shell, the system will reboot."
      echo
      PS1="(Repair filesystem) \#"; export PS1
      sulogin
    else # With an error code of 2 or 3, reboot the machine automatically:
      echo
      echo "***********************************"
      echo "*** The filesystem was changed. ***"
      echo "*** The system will now reboot. ***"
      echo "***********************************"
      echo
    fi
    echo "Unmounting file systems."
    umount -a -r
    mount -n -o remount,ro /
    echo "Rebooting system."
    sleep 2
    reboot -f
  fi
  # Remount the root filesystem in read-write mode
  echo "Remounting root device with read-write enabled."
  mount -w -v -n -o remount /
  if [ $? -gt 0 ] ; then
    echo
    echo "Attempt to remount root device as read-write failed!  This is going to"
    echo "cause serious problems."
    echo
    echo "Please press ENTER to continue, then start looking for the problem. "
    read junk;
  fi
else
  echo "Testing root filesystem status:  read-write filesystem"
  echo
  echo "*** ERROR: Root partition has already been mounted read-write. Cannot check!"
  echo
  echo "For filesystem checking to work properly, your system must initially mount"
  echo "the root partition as read only."
  echo
fi # Done checking root filesystem

# Check all the non-root filesystems:
if [ ! -r /etc/fastboot ]; then
   echo "Checking non-root filesystems:"
   # FIXME:  The real fsck wrapper is broken at the moment.
   # Use this ugly hack instead.
   for BLOCKDEV in $(grep -o "^[^ ]*" /etc/fstab) ; do
      if [ -b "${BLOCKDEV}" ]; then
         FSTYPE="$(grep "${BLOCKDEV}" /etc/fstab | sed 's/^[^ ]* *[^ ]* *//; s/ .*//')"
         case "${FSTYPE}" in
            ext[234])  FSCKFLAGS="-y" ;;
            *) unset FSCKFLAGS ;;
         esac
         if [ -x /sbin/fsck."${FSTYPE}" ]; then
            /sbin/fsck."${FSTYPE}" ${FORCEFSCK} ${FSCKFLAGS} "${BLOCKDEV}"
         fi
      fi
   done
fi

# Check if fstab and mediadaemon configuration is the correct version for this hardware
if ! grep -q "Mass storage" /etc/fstab; then
    cp /etc/fstab /etc/fstab.new

    if have_emmc; then
        cat /etc/fstab_emmc >> /etc/fstab.new
        ln -sf /etc/mediadaemon_emmc.conf /etc/mediadaemon.conf
    elif find_ubi_volume_num_from_name data > /dev/null; then
        cat /etc/fstab_nandonly >> /etc/fstab.new
        rm -f /etc/mediadaemon.conf
    else
        cat /etc/fstab_none >> /etc/fstab.new
        rm -f /etc/mediadaemon.conf
    fi

    if find_ubi_volume_num_from_name noslog >/dev/null; then
        cat /etc/fstab_noslog >> /etc/fstab.new
    fi

    if [ "$PROCESSOR_FAMILY" = "mx6" -a "$DISPSIZE" != "none" ]; then
        # HFP-2228: Configure zram swap
        cat >>/etc/fstab.new <<EOF
/dev/zram0       none                   swap    swap            0 0
EOF

        MEM_TOTAL="$(sed -n 's/^MemTotal.* \([0-9]*\) .*/\1/p' /proc/meminfo)"
        cat >/etc/udev/rules.d/12-dynamic.rules <<EOF
KERNEL=="zram0", SUBSYSTEM=="block", DRIVER=="", ACTION=="add", ATTR{disksize}=="0", ATTR{disksize}="${MEM_TOTAL}k", RUN+="/sbin/mkswap \$env{DEVNAME}"
EOF
    fi

    if ls /var/log/packages/MFDApp-* >/dev/null 2>&1; then
        cat >>/etc/fstab.new <<EOF
tmpfs            /tmp/NOS/browser       tmpfs   noauto,size=25%,nosuid,nodev,uid=$(id -u nos),gid=$(id -g nos)  0 0
EOF

        cat >>/etc/sysctl.conf <<EOF

# HFP-2229: Set available memory to 600% of RAM for MFDApp, overcommiting by 600%.
vm.overcommit_memory = 2
vm.overcommit_ratio = 600
EOF
    fi

    sync
    mv /etc/fstab.new /etc/fstab
fi

# Mount usbfs:
if grep -wq usbfs /proc/filesystems; then
  if ! grep -wq usbfs /proc/mounts ; then
    if ! grep -wq usbfs /etc/fstab; then
      mount -v usbfs /proc/bus/usb -t usbfs
    else
      mount -v /proc/bus/usb
    fi
  fi
fi

# Mount non-root file systems in fstab, but not NFS or SMB
# because TCP/IP is not yet configured, and not proc or sysfs
# because those have already been mounted.  Also check that
# devpts is not already mounted before attempting to mount
# it.  With a 2.6.x or newer kernel udev mounts devpts.
echo "Mounting non-root local filesystems:"
if /bin/grep -wq devpts /proc/mounts ; then
  mount -a -v -t nonfs,nosmbfs,nocifs,noproc,nosysfs,nodevpts
else
  mount -a -v -t nonfs,nosmbfs,nocifs,noproc,nosysfs
fi

# If mount failed to attach any of our local file
# systems then there must be something horribly wrong.
# Run a hard fsck on any block device that isn't mounted.
if [ "$?" -ge 32 ]; then
   for BLOCKDEV in $(cut -f 1 -d '#' /etc/fstab | \
      grep -v swap | \
      awk '{print $1}' ) ;
   do
      if [ -b "${BLOCKDEV}" ]; then
         for MOUNTPT in $(grep "${BLOCKDEV}" /etc/fstab | \
            awk '{print $2}' ) ;
         do
            if ! mountpoint -q "${MOUNTPT}" ; then
              fsck -f -C -v -y "${BLOCKDEV}" > "/run/shm/fsck.$(basename ${BLOCKDEV})" 2>&1
              mount "${MOUNTPT}"
            fi
         done
      fi
   done
fi

# Expose platform features
if [ -x /etc/rc.d/rc.hal ]; then
  /etc/rc.d/rc.hal start
fi

sysconfig_init

# HFP-792: Display debug/sysconfig flags during boot
[ "$DISPSIZE" != "none" -a -x /usr/sbin/debugservices ] && /usr/sbin/debugservices get-sysconfig --to-fbdev

# Populate some stuff in /tmp
mkdir /run/dbus
mkdir -p /tmp/var/cache/hald
mkdir /tmp/var/tmp
mkdir /tmp/oprofile
chmod 0777 /tmp

# Set system time
if [ -x /etc/rc.d/rc.time ]; then
   /etc/rc.d/rc.time start
fi

# Launch any 'runonce' commands.
for RUNONCE in $(ls /etc/rc.d/runonce.d/* 2>/dev/null) ; do
   if [ ! -f "${RUNONCE}" ]; then
      # Leave any weird stuff alone.
      continue
   fi
   if [ -x "${RUNONCE}" ]; then
      # Clean up after successful completion.
      "${RUNONCE}" && rm -f "${RUNONCE}"
      sync
   fi
done
unset RUNONCE

# HFP-173: Output debug message if watchdog timeout detected.
cut -f2 /proc/navico_platform/reset_src 2>/dev/null |
  sed -ne 's/^WDOG-TOUT$/WARNING: Reset source \0 detected!/p'

# Handle factory reset request, init user partitions.
if [ "${FAMILYID}" != "Radar" ]; then
    case "${FAMILYID}" in
        Hatchetfish|Lisa)
            if [ "$CODENAME-$SUBFAMILYID" = "thor-1" ]; then
                factdefkeys="feature disabled"
            else
                # HFP-91: Check for "-" and "+" keys using both HID and PS2 codes.
                factdefkeys="86 87$\|^74 78$\|^4 41"
            fi
            ;;
        Cassius)
            factdefkeys="1 30"
            ;;
        Floyd*)
            factdefkeys="16+"
            ;;
        Norris)
            factdefkeys="1 96"
            ;;
        Kiwi|Repower|Pukeko)
            factdefkeys="103 108"
            ;;
        Hobart)
            factdefkeys="1"
            ;;
        Sylvester)
            factdefkeys="104 109"
            ;;
        Bluefin)
            factdefkeys="16+$\|86 87$\|^74 78$\|^4 41"
            ;;
        *)
            factdefkeys="74 78" # "-" and "+"
            ;;
    esac

    factory_reset_req=false
    if grep -qs "^${factdefkeys}$" /run/shm/bootkeys; then
        echo "User has requested factory defaults. Deleting user related partitions..."
        factory_reset_req=true
    fi

    init_user_partitions $factory_reset_req

    unset factdefkeys
    unset factory_reset_req
fi

# HFP-1193: Cleanup after update from internal storage.
rm -f /home/updater.conf

# Initialize MAC
init_macaddr

# Start loading modules
if [ -x /etc/rc.d/rc.modules ]; then
  {
    sysctl -qp /etc/sysctl.conf
    /etc/rc.d/rc.modules start
  } &
  RC_MODULES_PID=$!
fi

if [ -x /etc/rc.d/rc.syslog ]; then
  /etc/rc.d/rc.syslog start
fi

if sysconfig_enabled debuglog && [ -x /etc/rc.d/rc.debuglog ]; then
  /etc/rc.d/rc.debuglog start
fi

### bring up network interfaces
if [ -x /etc/rc.d/rc.net ]; then
  /etc/rc.d/rc.net start

  if sysconfig_enabled telnet; then
    /usr/sbin/telnetd

    # Start SSH and NTPD (if installed)
    [ -x /etc/rc.d/rc.sshd ] && /etc/rc.d/rc.sshd start
    [ -x /etc/rc.d/rc.ntpd ] && /etc/rc.d/rc.ntpd start
  fi
fi

if [ -x /etc/rc.d/rc.bluetooth ]; then
    /etc/rc.d/rc.bluetooth start
fi

### turn on and (optionally) reprogram companion microcontrollers (if any)
if [ -x /etc/rc.d/rc.stm32 ]; then
  {
    stdio_syslog_start "rc.stm32"

    if [ "$FAMILYID" = "Hatchetfish" ]; then
      # Output hf-sync log captured inside initramfs (where syslog is unavailable)
      cat /run/hf-sync.log

      if /etc/rc.d/rc.stm32 is_update_reqd; then
	MCU_UPD_REQD=1
      else
	MCU_UPD_REQD=0
      fi

      # Peer MAC address is also exchanged here - must be done after "rc.net start"
      if [ "$(/sbin/hf-sync xch_mcu_upd_reqd_mac $MCU_UPD_REQD)" != "0" ]; then
        /etc/rc.d/rc.stm32 update
        /sbin/hf-sync post_mcu_upd_sync
      fi

      if [ "$CODENAME" = "thor" ]; then
          # Thor requires both USB ports be permanently mapped to CPU_A.
          [ "$SUBFAMILYID" = "1" ] && echo "t 9 1 1" | mcu-util
      else
          # HFP-65: On CPU_A, configure one USB port for each CPU (top=CPU_A, bottom=CPU_B)
          [ "$SUBFAMILYID" = "1" ] && echo "t 9 0 1" | mcu-util

          # Allow Ctrl-Atl-F[1:4] to change USB port mapping between CPU_A/B.
          /sbin/hf-sync usb_switcher </dev/null >/dev/null 2>&1 &
      fi

    elif [ "$FAMILYID" = "Radar" -a "$SUBFAMILYID" != "2" ]; then
      #do nothing for Radar unless it's Kahawai, then run update
      :
    else
      /etc/rc.d/rc.stm32 start
    fi
  } &

  RC_STM32_PID=$!
fi

### turn on GPS
if [ -x /etc/rc.d/rc.igps ]; then
  {
    # On some products the iGPS and Sonar share the same UART for reprogramming.
    if [ "$FAMILYID" = "Norris" -o "$FAMILYID" = "Dubya" ]; then
      wait_pid "${RC_STM32_PID:-}"   "STM32 software upgrade" 120
    fi
    stdio_syslog_start "rc.igps"
    /etc/rc.d/rc.igps start
  } &
  RC_IGPS_PID=$!
fi

# If we've reenabled the serial console,
# move the ttyS0 symlink so NOSApp can't find it (and init will).
if grep -q "console=ttymxc0" /proc/cmdline; then
  mv /dev/ttyS0 /dev/ttymxc0
  ln -sf /dev/null /dev/ttyS0
else
  # Have init open a console prompt on the default VC.
  # (probably the dummy virtual console.)
  ln -sf /dev/null /dev/ttymxc0
  ln -sf /dev/console /dev/tty0
fi

### initialise GPIOs exposed to userspace
if [ -x /etc/rc.d/rc.gpio ]; then
  /etc/rc.d/rc.gpio start

  ### set LCD contrast (most 5" and 7" models)
  if [ -r /etc/NOS/lcd_contrast ]; then
    /etc/rc.d/rc.gpio lcd_contrast=$(head -n1 /etc/NOS/lcd_contrast)
  fi
fi

### Thor services
if [ "$CODENAME-$SUBFAMILYID" = "thor-1" ]; then
  # Thor CPU A N2K to websocket bridge
  [ -x /etc/rc.d/rc.websocket ] && /etc/rc.d/rc.websocket start
  # Thor websocket client
  [ -x /etc/rc.d/rc.webclient ] && /etc/rc.d/rc.webclient start
  # Thor http service
  [ -x /etc/rc.d/rc.lighttpd ] && /etc/rc.d/rc.lighttpd start
fi

if [ "$CODENAME-$SUBFAMILYID" = "thor-2" ]; then
  # Thor CPU B nmea183 serial to tcp bridge.
  [ -x /etc/rc.d/rc.nmeatotcp ] && /etc/rc.d/rc.nmeatotcp start
fi

### Set up touch service
[ -x /etc/rc.d/rc.touch ] && /etc/rc.d/rc.touch start

### Set default key repeat rates on any keyboard devices (delay/period in ms)
for kbd in /dev/input/gpiokeys /dev/input/keypad /dev/input/matrixkey; do
  [ -e $kbd ] && inpdev --device=$kbd --rptdly 500 --rptprd 125
done

### start D-bus
if [ -x /etc/rc.d/rc.messagebus ]; then
  /etc/rc.d/rc.messagebus start
fi

if [ "${FAMILYID}" != "Radar" ]; then
  ### start hal
  if [ -x /etc/rc.d/rc.hald ]; then
    /etc/rc.d/rc.hald start
  fi

  # Start NFS-related services.
  if [ -x /etc/rc.d/rc.rpc ]; then
    /etc/rc.d/rc.rpc start
  fi
  if [ -x /etc/rc.d/rc.nfsd ]; then
    /etc/rc.d/rc.nfsd start
  fi
fi

# Enable core-dump handling
[ -x /etc/rc.d/rc.navico-coredump ] && /etc/rc.d/rc.navico-coredump start
# Enable goldcard service
[ -x /etc/rc.d/rc.goldcard ] && /etc/rc.d/rc.goldcard start

if [ "$DISPSIZE" != "none" ]; then
    if [ "$CODENAME-$SUBFAMILYID" != "thor-1" ]; then
	# Start screen mirroring service
	if [ -x /etc/rc.d/rc.rtspd ]; then
	    /etc/rc.d/rc.rtspd start
	fi

	# Configure ksz8895 switch to QoS mode
	ETHSW_CFG=/usr/share/firmware/${FAMILYID}-ksz8895.tsv
	if [ -r "$ETHSW_CFG" ]; then
	    ksz8895 spi-start && ksz8895 program "$ETHSW_CFG"
	fi

	# Start remote control service
	if [ -x /etc/rc.d/rc.remotecontrold ]; then
	    /etc/rc.d/rc.remotecontrold start
	fi

	# Start avahi service
	if [ -x /etc/rc.d/rc.avahi ]; then
	    /etc/rc.d/rc.avahi start
	fi

	# Start httpd service
	if [ -x /etc/rc.d/rc.httpd ]; then
	    /etc/rc.d/rc.httpd start
	fi

	if [ -x /etc/rc.d/rc.vsftpd ]; then
	    /etc/rc.d/rc.vsftpd start
	fi

        # HFP-1994: CAN logging diagnostics
	if [ -x /etc/rc.d/rc.canlog ]; then
	    /etc/rc.d/rc.canlog start
	fi
    fi

    [ -x /usr/sbin/updatesplashscreen ] && /usr/sbin/updatesplashscreen &
    [ -x /usr/libexec/colour_depth.sh ] && /usr/libexec/colour_depth.sh set auto

elif [ "$FAMILYID" = "SonarHub" -o "$FAMILYID" = "Luigi" -o "$FAMILYID" = "Newton" -o "$FAMILYID" = "Yoshi" ]; then
    if [ -x /etc/rc.d/rc.vsftpd ] && sysconfig_enabled ftp_root; then
	/etc/rc.d/rc.vsftpd start
    fi

    if [ -x /etc/rc.d/rc.ri2client ]; then
	/etc/rc.d/rc.ri2client start
    fi
fi

# Background waiting, so we don't prevent login prompt
(
  wait_pid "${RC_MODULES_PID:-}" "modules to load"        10
  wait_pid "${RC_STM32_PID:-}"   "STM32 software upgrade" 120
  wait_pid "${RC_IGPS_PID:-}"    "IGPS software upgrade"  120

  ### Run the application initialization script
  if [ "${FAMILYID}" = "Radar" ]; then
    if [ -x /etc/rc.d/rc.radar.home ]; then
      /etc/rc.d/rc.radar.home
    fi
    if [ -x /etc/rc.d/rc.radar.app ]; then
      /etc/rc.d/rc.radar.app start
    fi
    if [ -x /etc/rc.d/rc.radar.ruserver ]; then
      /etc/rc.d/rc.radar.ruserver start
    fi
  else
    if [ -x /etc/rc.d/rc.app ]; then
      /etc/rc.d/rc.app start
    fi

    swapon -a
  fi
) &
