#!/bin/sh

# udev rule:
# KERNEL=="mxc_hdmi", RUN+="udev_hdmi_action.sh"

# indicator files for HFP-799
HDMI_HOME=/home/nos
HDMI_LEGACY=${HDMI_HOME}/hdmi_legacy_support
HDMI_MODE_NOTFOUND=${HDMI_HOME}/hdmi_mode_notfound

family_id=$(cut -f2 /proc/navico_platform/family_id)
hdmi_dev=/devices/soc0/soc/20e0000.hdmi_video
verbose=false

log() {
    [ "$verbose" = "true" ] && echo $@ >&2
}

fbdev() {
    local dev=$1
    local fb_name="$(cat $dev/fb_name)"

    for fb in /sys/class/graphics/fb* ; do
        if [ "$(cat $fb/name)" = "$fb_name" ]; then
            echo $fb | sed "s/.*\(fb[0-9]\+\)/\1/"
            chgrp users ${fb}/blank
            chmod 664 ${fb}/blank
            return
        fi
    done
}

minornum() {
    local dev=$1
    printf "%d" 0x$(stat -c %T $(readlink -f $dev))
}

isrocky() {
    /usr/libexec/isrocky.sh /sys/$hdmi_dev/edid
}

mode_sort_ascending() {
    sort -k 1.3n
}

res_x() {
    local mode=$1
    echo $mode | sed 's/[^0-9]*\([0-9]\+\)x\([0-9]\+\).*/\1/g'
}
res_y() {
    local mode=$1
    echo $mode | sed 's/[^0-9]*\([0-9]\+\)x\([0-9]\+\).*/\2/g'
}

mode_aspect_ratio() {
    local mode=$1
    local x=$(res_x $mode)
    local y=$(res_y $mode)
    # two decimal places
    echo $((x*100/y))
}

# sorted list of fb modes, with or without U: modes
list_fb_modes() {
    local fb=$1
    local include=${2:-0}

    # sort by: detailed, standard, vesa, then others
    if [ $include = 0 ]; then
	# don't include fb driver modes "U:"
	cat /sys/class/graphics/$fb/modes | grep -v "U:" | sort
    else
	# include fb driver modes "U:" last
	cat /sys/class/graphics/$fb/modes | tr [DSVU] [abce] | sort | tr [abce] [DSVU]
    fi
}

preferred_mode() {
    local fb=$1
    list_fb_modes $fb | grep "D:" | head -n 1
}

# list of modes with screens native aspect ratio
list_aspect_modes() {
    local fb=$1

    # determine screen's natural (?) aspect ratio
    mode=$(preferred_mode $fb)

    if [ "$mode" ]; then
	log "Screen preferred mode is: $mode"
	aspect=$(mode_aspect_ratio $mode);
	log "Screen aspect ratio is: $aspect"

	# list matching modes
	for mode in $(list_fb_modes $fb) ; do
	    [ $(mode_aspect_ratio $mode) = $aspect ] && echo $mode
	done
    else
	log "Could not determine preferred mode: failed to read EDID?"
    fi
}

find_best_mode() {
    local fb=$1
    local res=$2
    local resx=$(res_x $res)
    local resy=$(res_y $res)

    # try to find exact match
    log "Looking for exact $res mode"
    found_mode=$(list_fb_modes $fb | grep $res | head -n 1)

    if [ -z "$found_mode" ]; then
	# otherwise, try to find good aspect ratio for screen
	log "No exact mode found, looking for aspect mode"
	for mode in $(list_aspect_modes $fb | mode_sort_ascending) ; do
	    if [ $(res_x $mode) -ge $resx ] && [ $(res_y $mode) -ge $resy ] ; then
		found_mode=$mode
		log "Found aspect mode: $mode"
		break;
	    fi
	done
    fi

    if [ -z "$found_mode" ]; then
	# otherwise, try to find smallest mode that is large enough
	log "No aspect mode, looking for large enough"
	for mode in $(list_fb_modes $fb | mode_sort_ascending) ; do
	    if [ $(res_x $mode) -ge $resx ] && [ $(res_y $mode) -ge $resy ] ; then
		found_mode=$mode
		log "Found large enough mode: $mode"
		break;
	    fi
	done
    fi

    if [ "$found_mode" ]; then
	log "Best mode is: $mode"
    else
        found_mode=$(list_fb_modes $fb | head -n 1)
	log "Failed to find suitable mode, scaling to fit"
    fi
    echo $found_mode
}

show_logo() {
    local dest=$1
    local logo="/usr/share/images/hdmi-${2:-'fail'}.bmp"

    clear_fb $dest

    if [ -e "$logo" ]; then
        fbdraw -f /dev/$dest -i "$logo" -x 50% -y 50% >/dev/null
        echo 0 > /sys/class/graphics/$dest/blank
    fi
}

clear_fb() {
    local fb=$1
    local stride=$(cat /sys/class/graphics/$fb/stride)
    local lines=$(cut -d, -f2 /sys/class/graphics/$fb/virtual_size)
    dd if=/dev/zero of=/dev/$fb bs=$stride count=$lines > /dev/null 2>&1
}

start_mirror () {
    local src=$1
    local dest=$2
    local res=$(tr , x < /sys/class/graphics/$src/virtual_size)
    local bpp=$(cat /sys/class/graphics/$src/bits_per_pixel)

    log "Source mode is:" $(cat /sys/class/graphics/$src/mode)

    if isrocky ; then
	mode=$(list_fb_modes $dest 1 | grep $res | head -n 1)
	log "Is a rocky"
	log "Using mode: $mode"
    elif [ -e $HDMI_LEGACY ] ; then
	# HFP-799: cassius 1.1 behaviour
	mode=$(list_fb_modes $dest 1 | grep $res | head -n 1)
	log "Is not a rocky"
	log "HDMI legacy mode requested"
	log "Forcing mode: $mode"
    else
	mode=$(find_best_mode $dest $res)
	log "Is not a rocky"
	log "Using best mode: $mode"
    fi

    if [ -n "$mode" ]; then
        local hdmi_disable=/run/shm/hdmi-output-disable

        rm -f $HDMI_MODE_NOTFOUND
        echo $bpp > /sys/class/graphics/$dest/bits_per_pixel
        echo $mode > /sys/class/graphics/$dest/mode

        if [ -r "$hdmi_disable" ]; then
            fbrotate --stop /dev/$dest > /dev/null 2>&1
            show_logo $dest "$(cut -f1 <"$hdmi_disable")"
        else
            fbrotate --rotate 0 --ipu $IPU -o /dev/$src:/dev/$dest > /dev/null
            clear_fb $dest
        fi
    else
        # HFP-642
        # not a rocky & no suitable mode found: show logo
        # nice to have: echo "please connect a monitor that supports " $resolution
        mode=$(preferred_mode $dest)
        [ "$mode" ] && preferred_mode $dest > /sys/class/graphics/$dest/mode

        show_logo $dest

        log "No mode found, displaying logo on $dest"
        echo "Need:" $(cat /sys/class/graphics/$src/mode) > $HDMI_MODE_NOTFOUND
        echo "Have:" $(cat /sys/class/graphics/$dest/mode) >> $HDMI_MODE_NOTFOUND
    fi
}

probe_hdmi () {
  local state=none
  local src
  local dest

  if [ -e /dev/fb_primary ]; then
    src=fb$(minornum /dev/fb_primary)
  else
    src=fb0
  fi

  dest=$(fbdev /sys/$hdmi_dev)

  if [ -e /sys/$hdmi_dev/cable_state ]; then
    state=$(cat /sys/$hdmi_dev/cable_state)
  fi

  case $state in
    plugin)
      start_mirror $src $dest
      ;;

    plugout)
      log "No monitor attached"
      fbrotate --stop /dev/$dest > /dev/null 2>&1
      ;;
  esac
}


[ "$family_id" = "Cassius" -o "$family_id" = "Forefraz" -o "$family_id" = "Norris" ] || exit 0

if [ $(cut -f3 /proc/navico_platform/cpuinfo) = "solo" ]; then
    IPU=0
else
    IPU=1
fi

# called "manually" when changing bpp
if [ "$1" = "--probe" ]; then
    if [ "$2" = "--verbose" ]; then
	verbose=true
    fi
    probe_hdmi
# called from udev
elif [ $DEVPATH = $hdmi_dev ]; then
  case $ACTION in
    # add|change)
    # for add: see rc.screen_rotation
    change)
      probe_hdmi
    ;;
  esac
fi
