#!/bin/sh

TARGETUSERNAME=nos
HOMEDIR=/home/${TARGETUSERNAME}
BOOTMODE_FILE=${HOMEDIR}/boot_mode
FTP_DIR=${HOMEDIR}/NOS/userdata
UPDATER_DIR=${FTP_DIR}/updaters
CONFIG_PARTITION=/etc/NOS
MAP_PARTITION=/media/factorydata
PATHS_INI=/etc/skel/paths.ini
BURNINFLAG=/tmp/burnin
NETWORK_CAPS="cap_net_bind_service,cap_net_raw+ep"
DHCP_PROBE_APP=/usr/sbin/dhcp_probe
BUSYBOX_APP=/bin/busybox
OPENTFTP_APP=/usr/bin/opentftpd
PIDFILE=/run/rc.app.pid
NOS_RESOURCE=/usr/share/NOS/resources
WEBENGINE_ARGS="--expose-gc"

source /etc/rc.d/rc.common
source /lib/functions/ubi.sh
navico_env_init

APP_PARAMS="-plugin evdevhotplug"\
":dir=/dev/input"\
":mask=\"mouse.*\""\
":internalkey=/dev/input/by-keymap/internal"\
":externalkey=/dev/input/by-keymap/external"

if [ "none" = "$DISPSIZE" ]; then
    APP_PARAMS="-platform nulloutput $APP_PARAMS"
else
    if [ "$FAMILYID" = "Lisa" ]; then
        CURSOR_TOUT=0
    fi

    APP_PARAMS="-platform navicofb:${FRAMEBUFFER}${CURSOR_TOUT+:cursortimeout=$CURSOR_TOUT} $APP_PARAMS"
fi

case "${FAMILYID}" in
  Cougar|Bobo)
    APP_PARAMS="$APP_PARAMS -plugin tslib -plugin evdevtouch"
    ;;
  Hatchetfish|Lisa)
    APP_PARAMS="$APP_PARAMS -plugin evdevtouch"
    chmod a+rwx /run/mcu-skt
    chmod go+w /sys/class/graphics/fb${FRAMEBUFFER##*[^0-9]}/blank
    chmod a+rw /dev/i2c-2
    ;;
  Cassius|Norris)
    APP_PARAMS="$APP_PARAMS -plugin evdevtouch -plugin navicorotary:/dev/input/rotary"
    # TODO: The path to the ap_guard socket can be retrieved from $PATHS_INI
    chown root:users /run/apguard /run/apguard/ap_guard
    chmod g+rwx /run/apguard /run/apguard/ap_guard
    ;;
  Forefraz)
    APP_PARAMS="$APP_PARAMS -plugin evdevtouch -plugin navicorotary:/dev/input/rotary"
    chmod a+rwx /run/mcu-skt
    ;;
  Clinton|Floyd*|Porky|Dubya)
    APP_PARAMS="$APP_PARAMS -plugin evdevtouch"
    ;;
esac

hdmi_mode() {
    if [ -x /usr/libexec/hdmi_mode.sh ]; then
	if [ "$HDMI_SAFE_MODE" = 1 ]; then
	    /usr/libexec/hdmi_mode.sh safe
	    HDMI_SAFE_MODE=0 # once only
	else
	    # preferred or user mode
	    /usr/libexec/hdmi_mode.sh apply
	fi
    fi
}

abs_path() {
  # Convert the specified directory (relative to HOMEDIR) to an absolute path.
  local path=$1
  [ "${path:0:1}" == "/" ] || path="$HOMEDIR/$path"
  readlink -f "$path" 2>/dev/null || printf "%s\n" "$HOMEDIR"
}

SETTINGSINIPATH=$(abs_path $(get_key_value "SettingsIni" "$PATHS_INI" "$HOMEDIR"))

addlogentry() {
  echo "$@"
}

enabledebuglogin() {
  # Launch telnetd. It won't launch a second time, because it's bind to the port fails
  /usr/sbin/telnetd

  # Launch ssh
  [ -x /etc/rc.d/rc.sshd ] && /etc/rc.d/rc.sshd start
}

hatchetfish_touch_stop() {
  # Kill eeti driver that writes to the serial and messes with serial baud
  if [ "${FAMILYID}" = Hatchetfish -a -e /tmp/TouchPID ]; then
    kill $(head -n1 /tmp/TouchPID)
    rm /tmp/TouchPID
  fi
}

init_homedir() {
    # Initialize nos home directory and noslog partition, or recreate if restore defaults requested.
    # A full factory reset recreates the /home partition (refer rc.S)

    # Ensure all the files in /etc/skel exist in the target user's home directory
    local src=/etc/skel
    local restore_defaults=${1:-false}

    ( cd "$src" && find ) | {
        local modified

        if [ ! -d "$HOMEDIR" -o "$restore_defaults" = true ]; then
            local noslog_path=/home/nos/log
            local noslog_ubidev=

            if noslog_ubidev=$(find_ubi_volume_num_from_name noslog); then
                noslog_ubidev=/dev/$noslog_ubidev

                if mountpoint -q "$noslog_path"; then
                    umount "$noslog_path"
                fi
            fi

            rm -fr "$HOMEDIR"
            mkdir -p "$HOMEDIR"

            if [ -n "${noslog_ubidev:-}" ]; then
                mkdir -p "$noslog_path"
                ubiupdatevol "$noslog_ubidev" -t
                mount "$noslog_path"
            fi

            modified=true
        fi

        local ftp_own="nos:ftp"
        local ftp_mod="1775"

        if [ "$(stat -c '%U:%G:%a' "$FTP_DIR" "$UPDATER_DIR" 2>&1|uniq)" != "${ftp_own}:${ftp_mod}" ]; then
            rm -fr "$UPDATER_DIR"
            mkdir -p "$UPDATER_DIR"
            modified=true
        fi

        case "$CODENAME" in
        gilligan)
            local s5100_usr_dir="/usr/share/NOS/s5100-updater"
            local s5100_home_dir="${HOMEDIR}/NOS/userdata/S5100 Updaters"
            if [ -r "$s5100_usr_dir" -a ! -r "$s5100_home_dir" ]; then
                ln -s "$s5100_usr_dir" "$s5100_home_dir"
                modified=true
            elif [ ! -r "$s5100_usr_dir" -a -r "$s5100_home_dir" ]; then
                rm -rf "$s5100_home_dir"
                modified=true
            fi
            ;;
        esac

        if [ "$(readlink -f "$HOMEDIR/paths.ini")" != "$PATHS_INI" ]; then
            printf 'Recreating "%s" link\n' "$PATHS_INI"
            ln -sf "$PATHS_INI" "$HOMEDIR/paths.ini"
            modified=true
        fi

        while read; do
            local file=${REPLY#./}
            local dest="$HOMEDIR/$file"

            if [ ! -e "$dest" -a ! -L "$dest" ]; then
                cp -av "$src/$file" "$dest"
                printf "init_homedir() copied '%s' -> '%s'\n" "$src/$file" "$dest"
                modified=true
            fi
        done

        if [ "${modified:-}" == "true" ]; then
            chown -R $TARGETUSERNAME:$(id -g $TARGETUSERNAME) "$HOMEDIR"
            chown "$ftp_own" "$FTP_DIR" "$UPDATER_DIR"
            chmod "$ftp_mod" "$FTP_DIR" "$UPDATER_DIR"
            sync
        fi
    }
}

check_updater_dir() {
  # HFP-1193: Prepare for internal-storage update if there is exactly one auto.upd present.
  local upd_count

  if upd_count=$(find "${UPDATER_DIR}/" -maxdepth 1 -type f -iname '*.auto.upd' | wc -l) && [ $upd_count -eq 1 ]
  then
    local symlink="/home/.upds"
    local relpath="${UPDATER_DIR#/home/}"

    rm -fr "$symlink" && \
    ln -svf "$relpath" "$symlink" && \
    echo 'EXTRADEVS=ubi0:home' > /home/updater.conf
  fi
}

read_boot_mode() {
  # Boot mode:
  #   1	TestApp
  #   2	NOSApp / normal boot (default)
  #   3	Burn-in
  #   4 Calibrate touch screen
  BOOTMODE=$(grep -s '^[1-4]$' "$BOOTMODE_FILE" || echo 2)
  if [ "$BOOTMODE" = 2 -a -f "$BOOTMODE_FILE" ]; then
    rm -f "$BOOTMODE_FILE"
    sync
  fi
}

read_boot_keys() {
  local KEYS=$(get_bootkeys_once)

  if [ -n "$KEYS" ]; then
    case "${FAMILYID}" in
    Cougar|Bobo)
      case "$KEYS" in
      "74") # Pg Down key - TestApp
        BOOTMODE=1
        ;;
      "25")  # PAGES key - soft reset (delete Settings.ini)
        rm "${SETTINGSINIPATH}/Settings.ini"
        ;;
      "45")  # WPT key - Touch screen calibration
        BOOTMODE=4
        ;;
      esac
      ;;

    Porky)
      case "$KEYS" in
      "74") # Pg Down key - TestApp
        BOOTMODE=1
        ;;
      "25")  # PAGES key - soft reset (delete Settings.ini)
        rm "${SETTINGSINIPATH}/Settings.ini"
        ;;
      "45")  # WPT key - Touch screen calibration
        BOOTMODE=4
        ;;
      esac
      ;;

    Clinton|Norris|Dubya)
      case "$KEYS" in
      "108") # Down key - TestApp
        BOOTMODE=1
        ;;
      "25")  # PAGES key - soft reset (delete Settings.ini)
        rm "${SETTINGSINIPATH}/Settings.ini"
        ;;
      "45") # Waypoint key - touch calibration
        BOOTMODE=4
        ;;
      esac
      ;;

    Hatchetfish)
      # Keys are returned *without* leading zeros on Hatchetfish
      # (Leading zeros imply octal format, which is wrong - values are decimal)
      case "$KEYS" in
      "108"|"42") # Down or Left-Shift key - TestApp
        BOOTMODE=1
        ;;
      "25")  # PAGES key - soft reset (delete Settings.ini)
        rm -f "${SETTINGSINIPATH}/Settings.ini"
        ;;
      "6 41"|"45 125"|"45 126") # op40:Win+X, kbd:left-Win-X, kbd:right-Win-X: 640x480 "safe mode"
	HDMI_SAFE_MODE=1
        ;;
      esac
      ;;

    Cassius)
      # Keys values are in decimal
      case "$KEYS" in
      "28") # Rotary press - TestApp
        BOOTMODE=1
        ;;
      "25")  # PAGES key - soft reset (delete Settings.ini)
        rm "${SETTINGSINIPATH}/Settings.ini"
        ;;
      "45") # Mark key - touch calibration
        BOOTMODE=4
        ;;
      esac
      ;;

    Forefraz)
      # Keys values are in decimal
      case "$KEYS" in
      "108"|"28") # Down key or rotary press - TestApp
        BOOTMODE=1
        ;;
      "1")  # Exit key - soft reset (delete Settings.ini)
        rm "${SETTINGSINIPATH}/Settings.ini"
        ;;
      esac
      ;;
    esac
  fi
}

launch_testapp() {
  local testapp=/usr/sbin/TestApp

  enabledebuglogin
  hatchetfish_touch_stop
  burn_in_cleanup

  # The backlight driver (LP8860) on Norris needs to be initialised
  /etc/rc.d/rc.gpio init-backlight

  # HFP-2087 to solve the SD card cannot be detected after TestApp restart
  killall -USR1 MediaDaemon

  mount -o remount,rw ${CONFIG_PARTITION}

  # Still run as root, but wrap inside su to set environment correctly.
  # Pass the location of paths.ini to TestApp so it has a chance to be
  # smarter about where to find things
  su -c "${DEBUGGER:+${DEBUGGER} }${testapp} $APP_PARAMS $PATHS_INI" -
  EXITCODE=$?

  sync
  mount -o remount,ro ${CONFIG_PARTITION}
}

launch_burn_in() {
  local burninclient=/usr/sbin/BurninClient

  enabledebuglogin
  hatchetfish_touch_stop

  #check map partition
  if [ -e $BURNINFLAG ]; then
    rm $BURNINFLAG
  else
    factory_partition_init "$MAP_PARTITION"
  fi

  mount -o remount,rw "$MAP_PARTITION"

  # BurninClient needs an IP address to function properly.  Spin.
  echo "BurninClient: Waiting for an IP address to appear on eth0."
  unset IPADDR
  while [ -z "${IPADDR}" ];
  do
    IPADDR=$(
	ifconfig | \
	grep -A1 '^eth0[^:]' | \
	grep "inet addr:" | \
	sed 's/.*inet addr://; s/ .*//'
    )
    sleep 1
  done
  unset IPADDR

  # Restart syslog daemons in factory mode
  /etc/rc.d/rc.syslog restart factory

  # Hatchetfish uses the dedicated nmea0183 rs422 port for comms
  local tty baudrate can ethernet
  if [ $FAMILYID = "Hatchetfish" ]; then
    if [ -e /dev/ttyS1 ]; then
      tty="--serial /dev/ttyS1"
      baudrate="--baudrate 4800"
    fi
  else
    if [ -e /dev/ttyS0 ]; then
      tty="--serial /dev/ttyS0"
      baudrate="--baudrate 4800"
    fi
  fi
  can="--can can0"
  ethernet="--ethernet-port 11079"

  if grep -q "^00:0e:c2" /sys/class/net/eth0/address ||
     grep -q "^00:0e:91" /sys/class/net/eth0/address ||
     sysconfig_enabled burnin_anymac
  then
    BURNINSERVERS="169.254.1.1:169.254.2.2:169.254.3.3:169.254.4.4:169.254.5.5"
    PRODUCT_ID=$(         get_key_value 'ProductId'        "$CONFIG_PARTITION/Config.ini")
    PRODUCT_REGION=$(     get_key_value 'ProductRegion'    "$CONFIG_PARTITION/Config.ini")
    FINAL_ASSY_PART_NUM=$(get_key_value 'FinalAssyPartNum' "$CONFIG_PARTITION/Config.ini")
    PRODUCT_DATASET=$(    get_key_value 'ProductDataset'   "$CONFIG_PARTITION/Config.ini")
    BURNIN_DATASET="${PRODUCT_ID}${PRODUCT_REGION}${PRODUCT_DATASET}"
    if [ -n "${FINAL_ASSY_PART_NUM}" ]; then
      TITLE="--title ${FINAL_ASSY_PART_NUM}"
      BURNIN_DATASET="${FINAL_ASSY_PART_NUM}"
    else
      FINAL_ASSY_PART_NUM="000-0000-00"
    fi

    # HFP-2360: TODO - Remove any existing fastresume data as this may cause torrents to halt and not resume.
    rm -rf /tmp/data/BurninClient
    # Still run as root, but wrap inside su to set environment correctly.
    # HFP-2224: XDG_DATA_HOME save restoredata in /tmp (instead of /root).
    su -c "XDG_DATA_HOME=/tmp ${DEBUGGER:+${DEBUGGER} }${burninclient} $APP_PARAMS $TITLE $tty $baudrate $can $ethernet \
      --dataset ${BURNIN_DATASET} \
      --serverlist ${BURNINSERVERS} \
      --partnum ${FINAL_ASSY_PART_NUM} \
      --dest /media/factorydata" -
  else
    su -c "${DEBUGGER:+${DEBUGGER} }${burninclient} $APP_PARAMS $tty $baudrate $can \
      --error --title Error --message \"Invalid MAC address\" --color red" -
  fi
  EXITCODE=$?

  sync
  mount -o remount,ro "$MAP_PARTITION"

  # Revert back to normal syslog mode
  /etc/rc.d/rc.syslog restart
}

burn_in_cleanup() {
  # Clean up any symlinks and torrent files on the chart partition.
  # Note that we only remount,rw if there are files to remove.
  (
    set -e	# Exit immediately if anything bad happens
    cd "$MAP_PARTITION"

    TMP_FILES=$(find . -maxdepth 1 \( -type l -o -name '*.torrent' \))

    if [ -n "$TMP_FILES" ]; then
      mount -o remount,rw "$MAP_PARTITION"
      printf "%s\n" "$TMP_FILES" | while read; do rm "$REPLY"; done
      mount -o remount,ro "$MAP_PARTITION"
    fi
  )
}

launch_touchcalibration() {
  /etc/rc.d/rc.touch fieldcalibration
  # Exit code 0 means next mode will always be 2=NOSApp
  EXITCODE=0
}

launch_thorapp() {
  local app_params="\
-platform navicofb:${FRAMEBUFFER}:cursortimeout=0 \
-plugin evdevhotplug:\
dir=/dev/input:\
mask=mouse.*:\
internalkeymap=/usr/share/navico-thor/resources/thor.qmap:\
externalkey=/dev/input/by-keymap/external:\
internalkey=/dev/input/by-keymap/internal"

  # THOR-409 determine ip address of other head
  local bail=20
  grep -f /run/peer-mac /proc/net/arp | cut -d' ' -f1 > /run/peer-ip
  while [ ! -s /run/peer-ip -a $bail -gt 0 ]; do
      if ! ping -w1 169.254.255.255 > /dev/null 2>&1 ; then
          # local network may not be up yet
          sleep 1
      fi
      grep -f /run/peer-mac /proc/net/arp | cut -d' ' -f1 > /run/peer-ip
      bail=$((bail-1))
  done

  mount -o remount,rw /media/factorydata

  # THOR-470 make default directories in factorydata
  local DIRS_FILE=/etc/thor.factorydirs
  if [ -f "${DIRS_FILE}" ]; then
      while read LINE; do
          if [ -n "${LINE}" ]; then
              mkdir -p /media/factorydata/${LINE}
              chmod 777 /media/factorydata/${LINE}
          fi
      done < ${DIRS_FILE}
  fi

  su -c "ecdis-thor $app_params" - ${TARGETUSERNAME}
  EXITCODE=$?
}

launch_nosapp() {
  local app=$(readlink -f /usr/bin/NOSApp)
  local launcher=/usr/bin/navico-applauncher
  local timeout=40

  #Ensure that CAPS has +ep at the end of the file and are in the same order as a getcap call on /usr/bin/NOSApp
  local caps="cap_net_bind_service,cap_net_admin,cap_net_raw,cap_sys_nice,cap_sys_time,cap_sys_tty_config+ep"

  set_app_caps "$app"		"$caps"
  set_app_caps "$launcher"	"$caps"

  # HFP-1253: set monitor resolution
  [ "$FAMILYID" = "Hatchetfish" -o "$FAMILYID" = "Lisa" ] && hdmi_mode

  # HFP-1158 Integrate Naviop services
  [ -x /etc/rc.d/rc.naviop ] && /etc/rc.d/rc.naviop start

  # HFP-864: Support for direct NOSApp logging has been removed.
  local APPLOGDIR=${HOMEDIR}/NOS/.log
  if [ -d "$APPLOGDIR" ]; then
    echo "WARNING: HFP-864: Deleting unsupported log directory $APPLOGDIR"
    rm -fr "$APPLOGDIR"
  fi

  local run_dir=$(get_key_value "RunDir" "$PATHS_INI" "/run/NOS")
  local tmp_dir=$(get_key_value "Temp"   "$PATHS_INI" "/tmp/NOS")
  local dir

  for dir in "$run_dir" "$tmp_dir"; do
    [ -d "$dir" ] || mkdir -p "$dir"
    chown $TARGETUSERNAME:users "$dir"
  done

  # HFP-1077 (see also HFP-868, HFP-949) let MediaDaemon know app has (re-)started
  killall -USR1 MediaDaemon

  # Use inotifyd (busybox) to detect files dropped into $DROP_DIR.  Files that pass
  # the given criteria (refer below) are moved into $CONFIG_PARTITION/features.
  local drop_dir=${tmp_dir}/features
  local inotify_cmd="/sbin/inotifyd - ${drop_dir}"
  mkdir -p ${drop_dir} &&
  chmod a+rw ${drop_dir} &&
  ${inotify_cmd} | while read EVENT DIR FILE; do
    [ "${EVENT}" = "w" ] &&		# w event = "Writable file is closed"
    [ "${DIR}" = "${drop_dir}" ] &&	# Sanity check (should always be true)
    [ "${FILE:-}" != "" ] &&		# Ignore changes to the directory itself
    [ "${FILE##[0-9]*}" = "" ] &&	# File name only contains digits
    [ ${FILE} -le 255 ] &&		# File name is 0-255
    cd ${DIR} &&
    {
      [ $(/bin/stat -c'%s' ${FILE}) -le 1024 ] && # File size <= 1KiB
      mv ${FILE} ${FILE}.$$ &&		# Make it ours
      sync &&				# Minimise remount-ro time (implied sync, below)
      mount -o remount,rw ${CONFIG_PARTITION} &&
      {
	mkdir -p ${CONFIG_PARTITION}/features &&
	mv ${FILE}.$$ ${CONFIG_PARTITION}/features/${FILE};
	mount -o remount,ro ${CONFIG_PARTITION};
      }
      cd -;
    }
  done >/dev/null 2>&1 &

  # Optionally including keymap parameters if the keymap file exists.
  local app_args=$APP_PARAMS
  local keymap_file=$(find "$NOS_RESOURCE" -maxdepth 1 -name '*.qmap' | head -n 1)

  if [ -n "$keymap_file" ]; then
    local extra_args="internalkeymap=${keymap_file}:"
    app_args=${app_args/evdevhotplug/evdevhotplug:$extra_args}
  fi

  app_args="$app_args $WEBENGINE_ARGS"

  local navico_coredump_flag=$(
    source /etc/navico-coredump.conf >/dev/null 2>&1
    echo "$NAVICO_COREDUMP_FLAG"
  )

  local debug_libs debug_env
  if sysconfig_enabled gpu_debug; then
    # HFP-559: LD_LIBRARY_PATH/LD_PRELOAD have a similar effect but cannot be used.
    debug_libs="/lib/ld-linux.so.3 --library-path /usr/lib/debug_gpu "
    debug_env="MALLOC_CHECK_=3 MALLOC_PERTURB_=170 "
  fi

  # HFP-1671: Increase probability of NOSApp being killed during OOM event.
  echo 500 > /proc/self/oom_score_adj

  # Prepare the browser cache.
  if [ -r /usr/lib/qt/lib/libQt5WebEngine.so ]; then
    local browser_cache="$(get_key_value "BrowserCache" "$PATHS_INI")"

    if [ -n "$browser_cache" ] && ! mountpoint -q "$browser_cache"; then
      mkdir -p "$browser_cache" && mount "$browser_cache"
    fi
  fi

  if [ -n "${DEBUGGER:-}" ]; then
    set_app_caps "${DEBUGGER%% *}" "$caps"
    su -c "$DEBUGGER $app $app_args" - $TARGETUSERNAME

  else
    su -c "${debug_env}${launcher} --timeout ${timeout} --kill \
      --coredumpflag ${navico_coredump_flag} \
      --socket ${run_dir}/app_monitor_daemon \
      -- ${debug_libs}${app} $app_args" \
      - ${TARGETUSERNAME}
  fi
  EXITCODE=$?

  echo 0 > /proc/self/oom_score_adj
  pkill -f "${inotify_cmd}"
  check_updater_dir

  # navico-applauncher returns 0 when NOSApp crashes
  if [ $EXITCODE -eq 0 ]; then
    EXITCODE=128
  fi
}

launch_yoshiapp() {
  local app=$(readlink -f /usr/bin/NOSApp)
  local launcher=/usr/bin/navico-applauncher
  local timeout=40

  #Ensure that CAPS has +ep at the end of the file and are in the same order as a getcap call on /usr/bin/NOSApp
  local caps="cap_net_bind_service,cap_net_admin,cap_net_raw,cap_sys_nice,cap_sys_time,cap_sys_tty_config+ep"

  set_app_caps "$app"		"$caps"
  set_app_caps "$launcher"	"$caps"

  local run_dir=$(get_key_value "RunDir" "$PATHS_INI" "/run/NOS")
  local tmp_dir=$(get_key_value "Temp"   "$PATHS_INI" "/tmp/NOS")
  local dir

  for dir in "$run_dir" "$tmp_dir"; do
    [ -d "$dir" ] || mkdir -p "$dir"
    chown $TARGETUSERNAME:users "$dir"
  done

  local app_args=$APP_PARAMS

  local navico_coredump_flag=$(
    source /etc/navico-coredump.conf >/dev/null 2>&1
    echo "$NAVICO_COREDUMP_FLAG"
  )

  # HFP-1671: Increase probability of NOSApp being killed during OOM event.
  echo 500 > /proc/self/oom_score_adj

  local pids=""
  local app_instance

  # Start the multiple application instances.
  for app_instance in 0 1 2; do
  (
    local appargs_instance="$app_args --instance=$app_instance"

    # Instance 0 uses the original path to preserve compatibility for development and debugging
    local app_monitor_path="${run_dir}/app_monitor_daemon"
    if [ $app_instance != 0 ]; then
        app_monitor_path="${app_monitor_path}${app_instance}"
    fi

    local app_command="$launcher --timeout $timeout --kill \
                      --coredumpflag $navico_coredump_flag \
                      --socket $app_monitor_path \
                      -- $app $appargs_instance"

    # Only run the debugger on instance 0.
    if [ $app_instance = 0 ] && [ -n "${DEBUGGER:-}" ]; then
        set_app_caps "${DEBUGGER%% *}" "$caps"
        su -c "$DEBUGGER $app $appargs_instance" - $TARGETUSERNAME
    else
      # Run each application instance in its own network namespace.
      ip netns exec netns${app_instance} su -c "$app_command" - $TARGETUSERNAME
    fi
  )&
  pids="$pids $!"
  done

  # Wait and chack the pids of the processes we created. If one process exits or
  # crashes, use its exitcode for EXITCODE and terminate the other processes.
  checkAppPids() {
    for pid in $pids; do
      if ! kill -0 $pid 2>/dev/null; then
        # Extract the exit code.
        wait $pid
        EXITCODE=$?

        # Disable the signal trap.
        trap - CHLD

        # Kill all the other active app process trees.
        for pid2 in $pids; do
          if kill -0 $pid2 2>/dev/null; then
            killtree $pid2 
          fi
        done

        # Exit on the first process found so EXITCODE is not updated with
        # the exit code (143) from the processes terminated by killtree above.
        break
      fi
    done
  }
  # Set up the signal trap for the child processes and wait.
  trap checkAppPids CHLD
  wait

  echo 0 > /proc/self/oom_score_adj

  # navico-applauncher returns 0 when NOSApp crashes
  if [ $EXITCODE -eq 0 ]; then
    EXITCODE=128
  fi
}

launch_app() {
    case "${CODENAME}-${SUBFAMILYID}" in
        thor-1)  launch_thorapp ;;
        yoshi-*) launch_yoshiapp ;;
        *)       launch_nosapp ;;
    esac
}

# For TestApp exit codes, refer TestApp/Widgets/tSpecialModes.h
handle_exitcode()
{
  # BOOTMODE tells us what state we are in
  #   1 = Test Mode
  #   2 = Normal Mode
  #   3 = Burnin Mode
  #   4 = Calibrate touch screen
  # EXITCODE tells us what state to go to
  #   16 = Normal Mode
  #   17 = Demo Mode
  #   18 = Burnin Mode
  #   19 = Test Mode
  #   20 = Reboot (Immediately)
  #   21 = Power Off
  #   22 = App Restart
  #   23 = App Stop (Break from script)
  #    * = Reboot (Crash)
  # SYSCTRLCODE
  #   (Lower numbers have higher priority when exchanged between CPU_A/B on Hatchetfish)
  #   10 = Power off
  #   20 = Reboot
  #   30 = Stop (break from script)
  #   40 = Continue (restart App)

  SYSCTRLCODE=40  # Default action is to restart app (or transition BOOTMODE)

  case "$EXITCODE" in

  0|16) # Normal
    addlogentry "Normal Mode"
    if [ "$BOOTMODE" = 1 -o "$BOOTMODE" = 3 ]; then
      init_homedir true # recreate nos user directory to restore defaults.
    fi
    BOOTMODE=2
    ;;

  17) # Demo
    addlogentry "Demo Mode"
    rm -f "$BOOTMODE_FILE"; sync
    BOOTMODE=2		# TODO  Not used, revert to "Normal" mode instead.
    ;;

  18) # Burnin
    addlogentry "Burnin Mode"
    if [ "$BOOTMODE" != 3 ]; then
      # assume we are at LCD/PCBA test, don't slow down the production line
      touch $BURNINFLAG
    fi
    echo 3 > "$BOOTMODE_FILE"; sync
    BOOTMODE=3
    ;;

  19) # Test
    addlogentry "Test Mode"
    echo 1 > "$BOOTMODE_FILE"; sync
    BOOTMODE=1
    ;;

  20) # Reboot
    addlogentry "App Reboot Request"
    SYSCTRLCODE=20
    ;;

  21) # Power Off
    addlogentry "App Power Off Request"
    SYSCTRLCODE=10
    ;;

  22) # Restart
    addlogentry "App Restart Request"
    # HFP-2912 Restart mediadaemon to make sure chart sharing available
    /etc/rc.d/rc.mediadaemon restart
    ;;

  23) # Stop (break from script)
    addlogentry "App Stop Request"
    SYSCTRLCODE=30
    ;;

  *) # Crash? Signals are 128+SIGNUM (eg 143 for SIGTERM)
    addlogentry "App Unhandled Exit Code: $EXITCODE"
    if sysconfig_enabled crash_halt; then
      SYSCTRLCODE=30 # Stop (break from script)
    else
      SYSCTRLCODE=20 # Reboot
    fi
    ;;

  esac
}

showlogo() {
    local logo=$1
    local xpos=$2
    local ypos=$3

    fbdraw -i $logo -x $xpos -y $ypos
}

showlogos() {
    if [ -e "$FRAMEBUFFER" ]; then
        local edge_offset=28
        local logo_gap=$((edge_offset/2))
        local xpos=$edge_offset # left offset
        local ypos=-$edge_offset # Bottom offset

        local resourcespath=$(get_key_value "Resources" "$PATHS_INI" "/usr/share/NOS/resources")
        local fbdraw_out=

        if grep -qis "^WeatherEnabled=true" "$SETTINGSINIPATH/Settings.ini"; then
            local dataset=$(get_key_value "ProductDataset" "$CONFIG_PARTITION/Config.ini" "undefined")
            case "$dataset" in
            Polaris)
                fbdraw_out="$(showlogo "$resourcespath/weather/sirius_weather_logo2.bmp" $xpos $ypos)"
                ;;
            *)
                fbdraw_out="$(showlogo "$resourcespath/weather/sirius_weather_logo.bmp" $xpos $ypos)"
                ;;
            esac
        fi

        if /etc/rc.d/rc.naviop installed; then
            if [ -n "$fbdraw_out" ]; then
                local prevlogo_h="$(echo "$fbdraw_out" | cut -f9)"
                ypos=$(( ypos>=0 ? ypos+logo_gap+prevlogo_h : ypos-logo_gap-prevlogo_h ))
            fi
            showlogo "$resourcespath/naviop/naviop_boot_logo.bmp" $xpos $ypos > /dev/null
        fi
    fi
}

init_config_ini() {
    local codename=$CODENAME

    # HFP-1388: Special handling for forefraz. 9"/16" have different ProductID's.
    case "${codename}-${SUBFAMILYID}" in
    foreman-0|frazier-0)
        codename="${codename}mini"
        ;;
    frazier-1)
        codename="${codename}maggie"
        ;;
    esac

    local new_config
    for new_config in "${NOS_RESOURCE}/Config-${codename}-${SUBFAMILYID}.ini" \
                      "${NOS_RESOURCE}/Config-${codename}.ini"; do
        if [ -s "$new_config" ]; then
            mount -o remount,rw "$CONFIG_PARTITION"
            local config="${CONFIG_PARTITION}/Config.ini"

            if grep -qi 'FactoryConfigIni=true' "$new_config"; then
                cat "$new_config" > "$config"
            else
                # HFP-1662: Mark Config.ini as being a default (rather than factory) version.
                cat "$new_config" - <<CONFIGINI_FOOTER >"$config"

[ProductDebug]
DefaultConfigIni=$codename
CONFIGINI_FOOTER
            fi

            chmod 644 "$config"
            sync
            mount -o remount,ro "$CONFIG_PARTITION"
            break
        fi
    done
}

start() {
  if [ -s "$PIDFILE" ]; then
    echo "$0: Already running" >&2
    return 0
  fi

  echo $$ >"$PIDFILE"

  stdio_syslog_start

  # Make a default config.ini if one does not exist
  [ -s $CONFIG_PARTITION/Config.ini ] || init_config_ini

  # HFP-1554: Frazier has different demo scripts for 9" and 16".
  # Create a symlink to the correct demo folder.
  case "$CODENAME" in
  foreman|frazier)
    demo_link_dir=$MAP_PARTITION/simulator/Demo
    if [ "$SUBFAMILYID" = "0" ]; then
      demo_dir=${demo_link_dir}-9
    else
      demo_dir=${demo_link_dir}-16
    fi
    if [ -d "$demo_dir" ] && [ ! -h "$demo_link_dir" ]; then
      mount -o rw,remount $MAP_PARTITION
      ln -sfn $demo_dir $demo_link_dir
      mount -o ro,remount $MAP_PARTITION
    fi
    ;;
  esac


  set_app_caps "$DHCP_PROBE_APP"	"$NETWORK_CAPS"
  set_app_caps "$BUSYBOX_APP"		"$NETWORK_CAPS"
  set_app_caps "$OPENTFTP_APP"		"$NETWORK_CAPS"

  init_homedir

  showlogos

  # HFP-949 Ensure MediaDaemon is running for all applications
  /etc/rc.d/rc.mediadaemon restart

  (
    cd "$HOMEDIR"

    read_boot_mode
    read_boot_keys

    while true; do
      if [ "$FAMILYID" = "Hatchetfish" ]; then
	local APP_WRAPPER KILL_DELAY
	local EXITCODE_FILE=/run/rc.app.exitcode.$$

	# Boot mode 2 (App) has lowest priority, so it must be mangled during the sync/exchange.
	# The other modes (1=TestApp, 3=Burn-in, 4=Touch cal) are already in priority order.
	[ "$BOOTMODE" = "2" ] && BOOTMODE=10
	BOOTMODE=$(hf-sync xch_app_mode $BOOTMODE)

	# HFP-78: Some delay is required for ATE synchronisation in the factory. KILL_DELAY is in ms.
	while true; do
	  case "$BOOTMODE" in
	    1)	APP_WRAPPER=launch_testapp; 		KILL_DELAY=5000 ;;
	    3)	APP_WRAPPER=launch_burn_in;		KILL_DELAY=5000 ;;
	    4)	APP_WRAPPER=launch_touchcalibration;	KILL_DELAY=0 ;;
	    2|*)APP_WRAPPER=launch_app;		        KILL_DELAY=0 ;
		BOOTMODE=2
		;;
	  esac

	  {
	    $APP_WRAPPER
	    echo $EXITCODE > $EXITCODE_FILE
	  } &
	  local APP_WRAPPER_PID=$!

	  local HDMI_HOTPLUG_PID=""
	  if [ -x /usr/sbin/hdmi-hotplug-modesetter ]; then
	    # hdmi-hotplug-modesetter will only exit if it changes the screen resolution
	    # in response to an HDMI hotplug event.
	    /usr/sbin/hdmi-hotplug-modesetter && killtree $APP_WRAPPER_PID &
	    HDMI_HOTPLUG_PID=$!
	  fi

	  local PEER_EXITED=false HDMI_HOTPLUG=false

	  if hf-sync app_monitor $APP_WRAPPER_PID $KILL_DELAY; then
	    # app_monitor killed the (local) app in response to the app exiting on the other CPU
	    PEER_EXITED=true
	  fi

	  if [ -n "$HDMI_HOTPLUG_PID" ]; then
	    if kill -0 $HDMI_HOTPLUG_PID 2>/dev/null; then
	      # hdmi-hotplug-modesetter still running, so app exited by itself
	      killtree $HDMI_HOTPLUG_PID
	    else
	      HDMI_HOTPLUG=true
	    fi
	  fi

	  if $PEER_EXITED || $HDMI_HOTPLUG; then
	    dd if=/dev/zero of=$FRAMEBUFFER 2>/dev/null # Clear the screen
	  fi

	  if $PEER_EXITED; then
	    addlogentry "Peer CPU exited application, resynchronising..."
	    BOOTMODE=2		# Lowest priority mode - let the other CPU choose the next mode
	    SYSCTRLCODE=40	# Lowest priority code - restart app or transition mode
	    rm -f "$BOOTMODE_FILE"; sync

	  elif $HDMI_HOTPLUG; then
	    addlogentry "HDMI hotplug changed screen resolution - performing local application restart"
	    continue

	  else
	    local PREV_BOOTMODE=$BOOTMODE
	    EXITCODE=$(cat $EXITCODE_FILE || echo "-1")
	    rm -f $EXITCODE_FILE

	    handle_exitcode

	    if [ "$PREV_BOOTMODE" = "$BOOTMODE" -a $SYSCTRLCODE -eq 40 ]; then
	      addlogentry "Local (unsynchronised) application restart ($BOOTMODE)"
	      continue
	    fi
	  fi

	  break
	done

	SYSCTRLCODE=$(FORCE_SYNC=true hf-sync xch_sys_ctrl $SYSCTRLCODE)

      else
	case "$BOOTMODE" in
	  1)	launch_testapp ;;
	  3)	launch_burn_in ;;
	  4)	launch_touchcalibration ;;
	  2|*)	launch_app ;;
	esac

	handle_exitcode
      fi

      # TODO The pre-poweroff/reboot code would be better placed in rc.R, except we can't differentiate between poweroff and reboot there
      case "$SYSCTRLCODE" in
	10)	# Power off
		# Turn off PoE (t 15 0 0) and power off the STM32 (t 20).
		[ "$FAMILYID" = "Hatchetfish" ] && echo -e "t 15 0 0\nt 20" | mcu-util
		/sbin/poweroff;	break ;;
	20)	# Reboot
		# Ensure both USB ports are mapped to CPU_A
		[ "$FAMILYID" = "Hatchetfish" ] && echo "t 9 1 1" | mcu-util
		/sbin/reboot; break ;;
	30)	# Exit (break from script)
		break ;;
	40|*)	# Continue (restart App)
		;;
      esac
    done

    rm -f "$PIDFILE"
  ) &

  echo $! >"$PIDFILE"
}


case ${1:-} in
    start)
        start
        ;;
    stop)
        stop_service
	[ -x /etc/rc.d/rc.naviop ] && /etc/rc.d/rc.naviop stop
        ;;
    restart)
        stop_service
        start
        ;;
    debug)
        DEBUGGER="/usr/bin/gdbserver host:${2:-1234}"
        start
        ;;
    *)
        echo "Usage: $0 {start | stop | restart | debug [port]}"
        ;;
esac
